<?php
/*
Plugin Name: Medienattribute Admin
Description: Admin-Tool: Bilder/Medien analysieren (Alt/Titel/Beschreibung), Verwendungen (Gutenberg/Elementor/Featured Image), Suche + Highlight, Inline-Edit, CSV Export, Löschen ungenutzter Bilder (v2.3.8 FINAL, robust).
Version: 2.3.8
Author: Karl-Heinz Rauch
License: GPLv2 or later
*/

if ( ! defined('ABSPATH') ) { exit; }

define('FRBE_SLUG', 'fr-bildattribute-export');

/* Admin Menu */
add_action('admin_menu', function () {
    add_menu_page(
        'Medienattribute Admin',
        'Medienattribute Admin',
        'manage_options',
        FRBE_SLUG,
        'frbe_render_admin_page',
        'dashicons-format-image',
        80
    );
});

/**
 * FINAL FIX: Elementor Pro "Notes" JS crash (window.top.$e undefined)
 * Dequeue ONLY on our plugin admin page to prevent global admin JS breakage.
 */
add_action('admin_enqueue_scripts', function () {
    if ( empty($_GET['page']) || $_GET['page'] !== FRBE_SLUG ) return;

    global $wp_scripts;
    if ( empty($wp_scripts) || empty($wp_scripts->registered) ) return;

    foreach ($wp_scripts->registered as $handle => $obj) {
        if ( empty($obj->src) ) continue;
        if ( strpos($obj->src, 'elementor-pro/assets/js/notes/') !== false ) {
            wp_dequeue_script($handle);
            wp_deregister_script($handle);
        }
    }
}, 9999);

/* Enqueue (only on our page) */
add_action('admin_enqueue_scripts', function($hook){
    if ( empty($_GET['page']) || $_GET['page'] !== FRBE_SLUG ) return;
    if ( ! current_user_can('manage_options') ) return;

    wp_enqueue_script('jquery');

    $cfg = array(
        'ajaxurl'        => admin_url('admin-ajax.php'),
        'nonce_edit'     => wp_create_nonce('frbe_inline_edit'),
        'nonce_delete'   => wp_create_nonce('frbe_delete'),
        'include_drafts' => (isset($_GET['frbe_drafts']) && $_GET['frbe_drafts'] === '1') ? 1 : 0,
    );

    wp_add_inline_script('jquery', 'window.FRBE_CFG = ' . wp_json_encode($cfg) . ';', 'before');
    wp_add_inline_script('jquery', frbe_js_bundle(), 'after');
    wp_add_inline_style('wp-admin', frbe_css_bundle());
});

function frbe_css_bundle(){
    return '
.frbe-searchbar{display:flex;gap:10px;align-items:center;margin:12px 0 14px;flex-wrap:wrap}
.frbe-searchbar input[type="search"]{min-width:380px}
.frbe-toggle{display:inline-flex;align-items:center;gap:6px;padding:4px 8px;border:1px solid #c3c4c7;border-radius:6px;background:#fff;white-space:nowrap;width:auto}
.frbe-toggle input{margin:0}
.frbe-debug-badge{display:inline-block;padding:2px 8px;border-radius:999px;background:#f0f0f1;border:1px solid #c3c4c7;font-size:12px}
.frbe-notice .notice{margin:0 0 12px 0}
.frbe-editable{min-width:140px}
.frbe-desc{min-width:240px}
.frbe-editable.editing{outline:2px solid #2271b1;padding:6px;background:#fff}
.frbe-status{font-size:12px;margin-left:8px}
.frbe-status.ok{color:#1d7f2a}
.frbe-status.err{color:#b32d2e}
.frbe-actions button{margin-right:6px}
.frbe-hit{background:#fff3b0 !important}
.frbe-hit-current{outline:3px solid #d63638 !important}
.frbe-pulse{box-shadow:0 0 0 4px rgba(214,54,56,0.25) !important}
button.frbe-delete{cursor:pointer}

/* Custom confirm modal */
#frbe-confirm{display:none; position:fixed; inset:0; background:rgba(0,0,0,.35); z-index:100000}
#frbe-confirm .frbe-confirm-card{background:#fff; max-width:520px; margin:10vh auto; padding:16px; border-radius:10px; box-shadow:0 10px 30px rgba(0,0,0,.25)}
#frbe-confirm .frbe-confirm-actions{display:flex; gap:10px; justify-content:flex-end; margin-top:12px}
';
}

function frbe_js_bundle(){
return <<<JS
(function($){
  if(!window.jQuery) return;

  const cfg = window.FRBE_CFG || {};
  const ajaxurl = cfg.ajaxurl || '';
  const nonceEdit = cfg.nonce_edit || '';
  const nonceDelete = cfg.nonce_delete || '';

  function escHtml(s){ return $('<div>').text(String(s||'')).html(); }

  function showNotice(type, msg){
    const \$wrap = $('#frbe-notice');
    if(!\$wrap.length) { alert(msg); return; }
    const cls = (type === 'success') ? 'notice-success' : (type === 'warning' ? 'notice-warning' : 'notice-error');
    \$wrap.html('<div class="notice '+cls+' is-dismissible"><p>'+ escHtml(msg) +'</p></div>');
  }

  function updateRowStatus(){
    const totalRows = $('#frbe-table tbody tr').length;
    const visibleRows = $('#frbe-table tbody tr:visible').length;
    $('#frbe-row-status').text(visibleRows + ' / ' + totalRows + ' Zeilen');
  }

  function setRowStatus(\$row, msg, ok){
    const \$s = \$row.find('.frbe-status');
    \$s.text(msg).removeClass('ok err').addClass(ok ? 'ok' : 'err');
    if(msg){ setTimeout(()=>{ \$s.text('').removeClass('ok err'); }, 2500); }
  }

  function collectRowData(\$row){
    return {
      id: \$row.data('aid'),
      alt: \$row.find('[data-field="alt"]').text().trim(),
      title: \$row.find('[data-field="title"]').text().trim(),
      desc: \$row.find('[data-field="desc"]').text().trim(),
    };
  }

  // Init
  $(function(){
    const count = $('#frbe-table tbody td.frbe-searchable').length;
    const delBtns = $('#frbe-table button.frbe-delete').length;
    $('#frbe-js-debug').text('JS aktiv ✓ | suchbare Zellen: ' + count + (delBtns ? (' | Delete-Buttons: ' + delBtns) : ''));
    updateRowStatus();
  });

  // Inline edit
  $(document).on('click', '.frbe-edit', function(e){
    e.preventDefault();
    const \$row = $(this).closest('tr');
    \$row.find('.frbe-editable').attr('contenteditable','true').addClass('editing');
    \$row.find('.frbe-save, .frbe-cancel').prop('disabled', false);
    \$row.find('.frbe-edit').prop('disabled', true);
    \$row.find('.frbe-editable').each(function(){ $(this).data('orig', $(this).text()); });
    setRowStatus(\$row,'Bearbeiten…',true);
  });

  $(document).on('click', '.frbe-cancel', function(e){
    e.preventDefault();
    const \$row = $(this).closest('tr');
    \$row.find('.frbe-editable').each(function(){
      const orig = $(this).data('orig');
      if(orig !== undefined) $(this).text(orig);
    });
    \$row.find('.frbe-editable').attr('contenteditable','false').removeClass('editing');
    \$row.find('.frbe-save, .frbe-cancel').prop('disabled', true);
    \$row.find('.frbe-edit').prop('disabled', false);
    setRowStatus(\$row,'Abgebrochen',true);
  });

  $(document).on('click', '.frbe-save', function(e){
    e.preventDefault();
    const \$row = $(this).closest('tr');
    const data = collectRowData(\$row);
    $(this).prop('disabled', true);
    setRowStatus(\$row,'Speichere…',true);

    $.post(ajaxurl, {
      action: 'frbe_inline_save',
      _ajax_nonce: nonceEdit,
      id: data.id,
      alt: data.alt,
      title: data.title,
      desc: data.desc
    }).done(function(resp){
      if(resp && resp.success){
        \$row.find('.frbe-editable').attr('contenteditable','false').removeClass('editing');
        \$row.find('.frbe-save, .frbe-cancel').prop('disabled', true);
        \$row.find('.frbe-edit').prop('disabled', false);
        setRowStatus(\$row,'Gespeichert ✅',true);
      } else {
        const msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Fehler beim Speichern';
        \$row.find('.frbe-save').prop('disabled', false);
        setRowStatus(\$row,msg,false);
      }
    }).fail(function(){
      \$row.find('.frbe-save').prop('disabled', false);
      setRowStatus(\$row,'AJAX Fehler',false);
    });
  });

  $(document).on('keydown', '.frbe-editable.editing', function(e){
    if(e.key === 'Enter' && !e.shiftKey){
      e.preventDefault();
      $(this).closest('tr').find('.frbe-save').trigger('click');
    }
    if(e.key === 'Escape'){
      e.preventDefault();
      $(this).closest('tr').find('.frbe-cancel').trigger('click');
    }
  });

  // Search
  let matches = [];
  let idx = -1;
  let rowHitSet = new Set();
  let showOnlyHitRows = false;

  function clearHighlights(){
    $('.frbe-hit').removeClass('frbe-hit');
    $('.frbe-hit-current').removeClass('frbe-hit-current');
    rowHitSet = new Set();
  }

  function updateSearchStatus(){
    const total = matches.length;
    const current = (idx >= 0) ? (idx + 1) : 0;
    $('#frbe-search-status').text(total ? (current + ' / ' + total) : '0 / 0');
    updateRowStatus();
  }

  function applyRowFilter(queryActive){
    const \$rows = $('#frbe-table tbody tr');
    if(showOnlyHitRows && queryActive){
      \$rows.each(function(){
        const aid = $(this).data('aid');
        $(this).toggle(rowHitSet.has(aid));
      });
    } else {
      \$rows.show();
    }
    updateRowStatus();
  }

  function goTo(i){
    if(!matches.length) return;
    idx = (i + matches.length) % matches.length;
    $('.frbe-hit-current').removeClass('frbe-hit-current');
    const el = matches[idx];
    const \$el = $(el);
    \$el.addClass('frbe-hit-current');
    el.scrollIntoView({behavior:'smooth', block:'center', inline:'nearest'});
    \$el.addClass('frbe-pulse');
    setTimeout(()=>{ \$el.removeClass('frbe-pulse'); }, 450);
    updateSearchStatus();
  }

  function build(q){
    clearHighlights();
    matches = [];
    idx = -1;

    q = (q || '').trim().toLowerCase();
    const queryActive = !!q;

    if(!q){
      applyRowFilter(false);
      updateSearchStatus();
      return;
    }

    $('#frbe-table tbody td.frbe-searchable').each(function(){
      const t = (this.textContent || '').toLowerCase();
      if(t.includes(q)){
        this.classList.add('frbe-hit');
        matches.push(this);
        rowHitSet.add($(this).closest('tr').data('aid'));
      }
    });

    applyRowFilter(queryActive);
    updateSearchStatus();
    if(matches.length) goTo(0);
  }

  function next(){ goTo(idx + 1); }
  function prev(){ goTo(idx - 1); }

  let timer = null;
  $(document).on('input', '#frbe-search', function(){
    clearTimeout(timer);
    const val = $(this).val();
    timer = setTimeout(()=>build(val), 150);
  });

  $(document).on('click', '#frbe-next', function(e){ e.preventDefault(); next(); });
  $(document).on('click', '#frbe-prev', function(e){ e.preventDefault(); prev(); });
  $(document).on('click', '#frbe-clear', function(e){
    e.preventDefault();
    $('#frbe-search').val('');
    build('');
  });

  $(document).on('keydown', '#frbe-search', function(e){
    if(e.key === 'Enter'){ e.preventDefault(); e.shiftKey ? prev() : next(); }
    if(e.key === 'Escape'){ e.preventDefault(); $('#frbe-search').val(''); build(''); }
  });

  $(document).on('change', '#frbe-onlyhits', function(){
    showOnlyHitRows = !!this.checked;
    build($('#frbe-search').val() || '');
  });

  // Custom confirm (no browser confirm)
  function frbeConfirm(){
    return new Promise((resolve) => {
      const wrap = document.getElementById('frbe-confirm');
      const yes = document.getElementById('frbe-confirm-yes');
      const no  = document.getElementById('frbe-confirm-no');

      if(!wrap || !yes || !no){
        resolve(window.confirm('Dieses Bild wird als „nicht verwendet“ erkannt. Wirklich endgültig löschen (inkl. Datei)?'));
        return;
      }

      const cleanup = () => {
        wrap.style.display = 'none';
        yes.onclick = null;
        no.onclick = null;
      };

      wrap.style.display = 'block';
      no.onclick = () => { cleanup(); resolve(false); };
      yes.onclick = () => { cleanup(); resolve(true); };
    });
  }

  // DELETE
  async function doDelete(btn, id){
    try{
      id = parseInt(id, 10);
      if(!id){
        showNotice('error','Ungültige ID.');
        return;
      }
      if(!nonceDelete){
        showNotice('error','Delete-Nonce fehlt. Seite neu laden.');
        return;
      }

      const ok = await frbeConfirm();
      if(!ok) return;

      if(btn){
        btn.disabled = true;
        btn.textContent = 'Lösche…';
      }

      const params = new URLSearchParams();
      params.append('action', 'frbe_delete_attachment');
      params.append('_ajax_nonce', nonceDelete);
      params.append('attachment_id', String(id));
      params.append('include_drafts', (cfg.include_drafts ? '1' : '0'));

      const res = await fetch(ajaxurl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: {'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'},
        body: params.toString()
      });

      const text = await res.text();
      let json = null;
      try { json = JSON.parse(text); } catch(e) {}

      if(json && json.success){
        showNotice('success', (json.data && json.data.message) ? json.data.message : 'Bild wurde gelöscht.');
        const tr = btn ? btn.closest('tr') : null;
        if(tr) tr.remove();
        updateRowStatus();
        return;
      }

      const msg = (json && json.data && json.data.message)
        ? json.data.message
        : ('Löschen fehlgeschlagen. HTTP ' + res.status);

      showNotice('error', msg);

      if(btn){
        btn.disabled = false;
        btn.textContent = 'Löschen';
      }
    } catch(err){
      showNotice('error','Exception beim Löschen: ' + (err && err.message ? err.message : err));
      if(btn){
        btn.disabled = false;
        btn.textContent = 'Löschen';
      }
    }
  }

  // ULTRA-ROBUST capture: only pointerup + lock to avoid multiple triggers
  (function bindDeleteUltra(){
    if (window.__FRBE_DEL_BOUND__) return;
    window.__FRBE_DEL_BOUND__ = true;

    let frbeDeleteBusy = false;

    function findDeleteButtonFromEvent(e){
      const path = (e && typeof e.composedPath === 'function') ? e.composedPath() : null;
      if (path && path.length){
        for (const el of path){
          if (el && el.nodeType === 1 && el.matches && el.matches('button.frbe-delete')) return el;
        }
      }
      const t = e && e.target;
      if (t && t.closest) return t.closest('button.frbe-delete');
      return null;
    }

    async function handler(e){
      const btn = findDeleteButtonFromEvent(e);
      if(!btn) return;

      const table = btn.closest ? btn.closest('#frbe-table') : null;
      if(!table) return;

      if (frbeDeleteBusy) return;
      frbeDeleteBusy = true;
      setTimeout(() => { frbeDeleteBusy = false; }, 800);

      e.preventDefault();
      e.stopPropagation();

      const id =
        btn.getAttribute('data-aid') ||
        (btn.closest('tr') ? btn.closest('tr').getAttribute('data-aid') : '');

      await doDelete(btn, id);
    }

    window.addEventListener('pointerup', handler, true);
    console.log('[FRBE] Delete ULTRA handler bound ✓');
  })();

  window.FRBE_Notice = showNotice;

})(jQuery);
JS;
}

/* AJAX: Inline save */
add_action('wp_ajax_frbe_inline_save', function () {
    if ( ! current_user_can('manage_options') ) wp_send_json_error(array('message' => 'Keine Berechtigung.'), 403);
    check_ajax_referer('frbe_inline_edit');

    $id = isset($_POST['id']) ? absint($_POST['id']) : 0;
    if ( ! $id ) wp_send_json_error(array('message' => 'Ungültige ID.'), 400);

    $post = get_post($id);
    if ( ! $post || $post->post_type !== 'attachment' ) wp_send_json_error(array('message' => 'Kein Attachment.'), 400);
    if ( ! current_user_can('edit_post', $id) ) wp_send_json_error(array('message' => 'Keine Berechtigung (edit_post).'), 403);

    $alt   = isset($_POST['alt'])   ? wp_strip_all_tags( wp_unslash($_POST['alt']) ) : '';
    $title = isset($_POST['title']) ? wp_strip_all_tags( wp_unslash($_POST['title']) ) : '';
    $desc  = isset($_POST['desc'])  ? wp_kses_post( wp_unslash($_POST['desc']) ) : '';

    update_post_meta($id, '_wp_attachment_image_alt', $alt);

    $res = wp_update_post(array(
        'ID'           => $id,
        'post_title'   => $title,
        'post_content' => $desc,
    ), true);

    if ( is_wp_error($res) ) wp_send_json_error(array('message' => $res->get_error_message()), 500);
    wp_send_json_success(array('message' => 'ok'));
});

/* AJAX: Delete attachment */
add_action('wp_ajax_frbe_delete_attachment', function () {
    if ( ! current_user_can('delete_posts') ) wp_send_json_error(array('message' => 'Keine Berechtigung.'), 403);
    check_ajax_referer('frbe_delete');

    $attachment_id  = isset($_POST['attachment_id']) ? absint($_POST['attachment_id']) : 0;
    $include_drafts = ! empty($_POST['include_drafts']) ? true : false;

    if ( ! $attachment_id ) wp_send_json_error(array('message' => 'Ungültige ID.'), 400);

    $post = get_post($attachment_id);
    if ( ! $post || $post->post_type !== 'attachment' ) wp_send_json_error(array('message' => 'Anhang nicht gefunden.'), 404);
    if ( ! current_user_can('delete_post', $attachment_id) ) wp_send_json_error(array('message' => 'Keine Berechtigung (delete_post).'), 403);

    $mime = get_post_mime_type($attachment_id);
    if ( ! $mime || strpos($mime, 'image/') !== 0 ) wp_send_json_error(array('message' => 'Anhang ist kein Bild.'), 400);

    $url = wp_get_attachment_url($attachment_id);

    $usages = frbe_find_attachment_usage($url, $attachment_id, $include_drafts, true);
    if ( ! empty($usages) ) wp_send_json_error(array('message' => 'Bild wird verwendet und wurde NICHT gelöscht.'), 409);

    $deleted = wp_delete_attachment($attachment_id, true);
    if ( ! $deleted ) wp_send_json_error(array('message' => 'wp_delete_attachment() ist fehlgeschlagen.'), 500);

    wp_send_json_success(array('message' => 'Bild wurde gelöscht.'));
});

/* CSV export */
add_action('admin_init', function () {
    if ( ! is_admin() || ! current_user_can('manage_options') ) return;
    if ( ! isset($_GET['page']) || $_GET['page'] !== FRBE_SLUG ) return;
    if ( ! isset($_GET['frbe_download']) ) return;

    $filter_unused  = isset($_GET['frbe_unused']) && $_GET['frbe_unused'] === '1';
    $filter_no_alt  = isset($_GET['frbe_no_alt']) && $_GET['frbe_no_alt'] === '1';
    $include_drafts = isset($_GET['frbe_drafts']) && $_GET['frbe_drafts'] === '1';

    $rows = frbe_get_image_rows($include_drafts);

    if ($filter_unused) $rows = array_filter($rows, fn($r) => trim($r['Verwendet_in']) === '');
    if ($filter_no_alt) $rows = array_filter($rows, fn($r) => trim((string)$r['Alt_Text']) === '');

    frbe_download_csv($rows);
});

/* Usage detection */
function frbe_find_attachment_usage( $attachment_url, $attachment_id = 0, $include_drafts = false, $strict = false ) {
    global $wpdb;

    if ( empty($attachment_url) && ! $attachment_id ) return array();
    $id = (int)$attachment_id;

    $statuses = array('publish','private');
    if ($include_drafts) $statuses = array_merge($statuses, array('draft','pending','future'));
    $status_in = "'" . implode("','", array_map('esc_sql', $statuses)) . "'";

    $post_types = array('post','page','product');
    $pt_in = "'" . implode("','", array_map('esc_sql', $post_types)) . "'";

    $likes = array();

    if ($attachment_url) {
        $likes[] = '%' . $wpdb->esc_like($attachment_url) . '%';
        if (!$strict) {
            $parsed = wp_parse_url($attachment_url);
            if (!empty($parsed['path'])) $likes[] = '%' . $wpdb->esc_like($parsed['path']) . '%';
        }
    }

    if ($id) {
        $likes[] = '%"id":' . $id . '%';
        $likes[] = '%wp-image-' . $id . '%';
        $likes[] = '%attachment_id="' . $id . '"%';
        $likes[] = "%attachment_id='" . $id . "'%";
    }

    if (empty($likes)) return array();

    $where_content = array();
    foreach ($likes as $l) $where_content[] = $wpdb->prepare("p.post_content LIKE %s", $l);

    $results = array();

    $sql_content = "
        SELECT p.ID, p.post_title
        FROM {$wpdb->posts} p
        WHERE p.post_status IN ($status_in)
          AND p.post_type IN ($pt_in)
          AND (" . implode(' OR ', $where_content) . ")
        LIMIT 500
    ";
    $r1 = $wpdb->get_results($sql_content);
    if ($r1) $results = array_merge($results, $r1);

    $meta_keys = array('_elementor_data','_elementor_page_settings');
    $meta_key_in = "'" . implode("','", array_map('esc_sql', $meta_keys)) . "'";

    $where_meta = array();
    foreach ($likes as $l) $where_meta[] = $wpdb->prepare("pm.meta_value LIKE %s", $l);

    $sql_meta = "
        SELECT DISTINCT p.ID, p.post_title
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON pm.post_id = p.ID
        WHERE p.post_status IN ($status_in)
          AND p.post_type IN ($pt_in)
          AND pm.meta_key IN ($meta_key_in)
          AND (" . implode(' OR ', $where_meta) . ")
        LIMIT 500
    ";
    $r2 = $wpdb->get_results($sql_meta);
    if ($r2) $results = array_merge($results, $r2);

    if ($id) {
        $sql_thumb = $wpdb->prepare("
            SELECT DISTINCT p.ID, p.post_title
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON pm.post_id = p.ID
            WHERE p.post_status IN ($status_in)
              AND p.post_type IN ($pt_in)
              AND pm.meta_key = '_thumbnail_id'
              AND pm.meta_value = %d
            LIMIT 500
        ", $id);
        $r3 = $wpdb->get_results($sql_thumb);
        if ($r3) $results = array_merge($results, $r3);
    }

    if (!$results) return array();

    $uniq = array();
    foreach ($results as $r) $uniq[(int)$r->ID] = $r;
    return array_values($uniq);
}

/* Rows */
function frbe_get_image_rows($include_drafts = false) {
    $attachments = get_posts(array(
        'post_type'      => 'attachment',
        'post_mime_type' => 'image',
        'posts_per_page' => -1,
        'post_status'    => 'inherit',
        'orderby'        => 'ID',
        'order'          => 'DESC',
    ));

    $rows = array();

    foreach ($attachments as $a) {
        $id   = (int)$a->ID;
        $url  = wp_get_attachment_url($id);
        $alt  = get_post_meta($id, '_wp_attachment_image_alt', true);
        $tit  = get_the_title($id);
        $desc = $a->post_content;

        $usages = frbe_find_attachment_usage($url, $id, $include_drafts, false);

        $usage_strings = array();
        if ($usages) foreach ($usages as $u) $usage_strings[] = $u->ID . ': ' . $u->post_title;

        $rows[] = array(
            'ID'           => $id,
            'Bild_URL'     => $url,
            'Alt_Text'     => $alt,
            'Titel'        => $tit,
            'Beschreibung' => $desc,
            'Verwendet_in' => implode(' | ', $usage_strings),
        );
    }

    return $rows;
}

/* CSV */
function frbe_download_csv($rows) {
    $rows = array_values((array)$rows);

    while (ob_get_level()) { ob_end_clean(); }

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=bildattribute-export.csv');
    header('Pragma: no-cache');
    header('Expires: 0');

    echo "\xEF\xBB\xBF";

    $fp = fopen('php://output', 'w');

    $write_line = function($handle, $fields) {
        foreach ($fields as $k => $v) {
            if (is_string($v)) $fields[$k] = preg_replace("/\r\n|\r|\n/", ' ', $v);
        }
        if (version_compare(PHP_VERSION, '8.1.0', '>=')) {
            fputcsv($handle, $fields, ';', '"', '\\', "\r\n");
        } else {
            $mem = fopen('php://temp', 'r+');
            fputcsv($mem, $fields, ';', '"', '\\');
            rewind($mem);
            $line = stream_get_contents($mem);
            fclose($mem);
            $line = rtrim($line, "\r\n") . "\r\n";
            fwrite($handle, $line);
        }
    };

    if (!empty($rows) && is_array($rows[0])) $write_line($fp, array_keys($rows[0]));
    foreach ($rows as $r) $write_line($fp, $r);

    fclose($fp);
    exit;
}

/* Admin page */
function frbe_render_admin_page() {
    if ( ! current_user_can('manage_options') ) return;

    $filter_unused  = isset($_GET['frbe_unused']) && $_GET['frbe_unused'] === '1';
    $filter_no_alt  = isset($_GET['frbe_no_alt']) && $_GET['frbe_no_alt'] === '1';
    $include_drafts = isset($_GET['frbe_drafts']) && $_GET['frbe_drafts'] === '1';

    $rows = frbe_get_image_rows($include_drafts);

    if ($filter_unused) $rows = array_filter($rows, fn($r) => trim($r['Verwendet_in']) === '');
    if ($filter_no_alt) $rows = array_filter($rows, fn($r) => trim((string)$r['Alt_Text']) === '');

    ?>
    <div class="wrap">
        <h1>Bildattribute Export</h1>

        <div id="frbe-notice" class="frbe-notice"></div>

        <!-- Custom confirm modal -->
        <div id="frbe-confirm" role="dialog" aria-modal="true" aria-labelledby="frbe-confirm-title">
          <div class="frbe-confirm-card">
            <h2 id="frbe-confirm-title" style="margin:0 0 8px;">Bild wirklich löschen?</h2>
            <p style="margin:0 0 12px;">Dieses Bild wird als „nicht verwendet“ erkannt. Wirklich endgültig löschen (inkl. Datei)?</p>
            <div class="frbe-confirm-actions">
              <button type="button" class="button" id="frbe-confirm-no">Abbrechen</button>
              <button type="button" class="button button-primary" id="frbe-confirm-yes">Ja, löschen</button>
            </div>
          </div>
        </div>

        <form method="get" style="margin-bottom:10px;">
            <input type="hidden" name="page" value="<?php echo esc_attr(FRBE_SLUG); ?>">
            <label style="margin-right: 1em;">
                <input type="checkbox" name="frbe_unused" value="1" <?php checked($filter_unused); ?>>
                Nur Bilder ohne Verwendung („Verwendet in” leer)
            </label>
            <label style="margin-right: 1em;">
                <input type="checkbox" name="frbe_no_alt" value="1" <?php checked($filter_no_alt); ?>>
                Nur Bilder ohne Alt-Text
            </label>
            <label style="margin-right: 1em;">
                <input type="checkbox" name="frbe_drafts" value="1" <?php checked($include_drafts); ?>>
                Auch Entwürfe berücksichtigen
            </label>
            <button class="button">Filter anwenden</button>
        </form>

        <div class="frbe-searchbar">
            <input id="frbe-search" type="search" class="regular-text"
                   placeholder="Suchen in ID, Bild-URL, Alt-Text, Titel, Beschreibung, Verwendet in …" />
            <button id="frbe-prev" class="button" type="button">◀︎ Vorheriger</button>
            <button id="frbe-next" class="button" type="button">Nächster ▶︎</button>
            <button id="frbe-clear" class="button" type="button">✕ Leeren</button>
            <span id="frbe-search-status">0 / 0</span>
            <span id="frbe-row-status">0 / 0 Zeilen</span>
            <label class="frbe-toggle" title="Blendet alle Zeilen ohne Treffer aus (nur wenn ein Suchbegriff aktiv ist).">
                <input id="frbe-onlyhits" type="checkbox">
                Nur Treffer anzeigen
            </label>
            <span class="frbe-debug-badge" id="frbe-js-debug">JS aktiv? …</span>
        </div>

        <p>
            <?php $csv_url = add_query_arg( array_merge($_GET, array('frbe_download'=>1)), admin_url('admin.php') ); ?>
            <a class="button button-primary" href="<?php echo esc_url($csv_url); ?>">CSV herunterladen</a>
        </p>

        <table id="frbe-table" class="widefat striped">
            <thead>
            <tr>
                <th>ID</th>
                <th>Preview</th>
                <th>Bild URL</th>
                <th>Alt-Text</th>
                <th>Titel</th>
                <th>Beschreibung</th>
                <th>Verwendet in</th>
                <th>Aktion</th>
            </tr>
            </thead>
            <tbody>
            <?php if (empty($rows)): ?>
                <tr><td colspan="8">Keine Bilder gefunden.</td></tr>
            <?php else: foreach ($rows as $row): ?>
                <?php
                    $aid = (int)$row['ID'];
                    $is_unused = (trim((string)$row['Verwendet_in']) === '');

                    $used_html = '';
                    $used_raw = trim((string)$row['Verwendet_in']);
                    if ($used_raw !== '') {
                        $parts = array_map('trim', explode('|', $used_raw));
                        $links = array();
                        foreach ($parts as $p) {
                            if (preg_match('/^(\\d+)\\s*:\\s*(.+)$/', $p, $m)) {
                                $pid = (int)$m[1];
                                $ptitle = $m[2];
                                $edit = get_edit_post_link($pid, 'raw');
                                $links[] = $edit ? '<a href="'.esc_url($edit).'">'.esc_html($pid.': '.$ptitle).'</a>' : esc_html($p);
                            } else {
                                $links[] = esc_html($p);
                            }
                        }
                        $used_html = implode(' | ', $links);
                    }
                ?>
                <tr data-aid="<?php echo esc_attr($aid); ?>">
                    <td class="frbe-searchable"><?php echo esc_html($row['ID']); ?></td>
                    <td><?php if (!empty($row['Bild_URL'])): ?><img src="<?php echo esc_url($row['Bild_URL']); ?>" style="max-width:70px;height:auto;"><?php endif; ?></td>
                    <td class="frbe-searchable" style="word-break:break-all;">
                        <?php if (!empty($row['Bild_URL'])): ?>
                            <a target="_blank" href="<?php echo esc_url($row['Bild_URL']); ?>"><?php echo esc_html($row['Bild_URL']); ?></a>
                        <?php endif; ?>
                    </td>
                    <td class="frbe-editable frbe-searchable" data-field="alt"><?php echo esc_html($row['Alt_Text']); ?></td>
                    <td class="frbe-editable frbe-searchable" data-field="title"><?php echo esc_html($row['Titel']); ?></td>
                    <td class="frbe-editable frbe-desc frbe-searchable" data-field="desc"><?php echo esc_html($row['Beschreibung']); ?></td>
                    <td class="frbe-searchable"><?php echo $used_html ? $used_html : '—'; ?></td>
                    <td class="frbe-actions">
                        <button class="button frbe-edit" type="button">Bearbeiten</button>
                        <button class="button button-primary frbe-save" type="button" disabled>Speichern</button>
                        <button class="button frbe-cancel" type="button" disabled>Abbrechen</button>
                        <span class="frbe-status"></span>

                        <?php if ($is_unused && current_user_can('delete_posts')): ?>
                            <div style="margin-top:6px;">
                                <button class="button button-secondary frbe-delete" type="button"
                                        data-aid="<?php echo (int)$aid; ?>">
                                    Löschen
                                </button>
                            </div>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <p style="margin-top:1em;color:#666;">
            Tipp: Enter im Suchfeld = nächster Treffer, Shift+Enter = vorheriger, ESC = leeren. Enter im Edit-Feld = speichern, ESC = abbrechen.
        </p>
    </div>
    <?php
}
